function [Data] = ConstantCurrent(Location, Folder1, FileNameEC, Area, AreaErr, VarName, RFileName, FA)

%{
This m-file belongs to the article 'The role of the electrolyte identity on
electrochemical CO2 conversion in the absence of mass transfer limitations'
Authors: Anne Sustronk (1,2) , Nieck E. Benes (1) , Guido Mul (2)
(1) Films in Fluids, Department of Science and Technology, University of 
Twente, Enschede, The Netherlands
(2) Photocatalytic Synthesis Group, Department of Science and Technology, 
University of Twente, Enschede, The Netherlands

The file is used to prepare Figures 3A and S2.

The next files are required to execute this file: CalH2Data.txt, 
CalCOData.txt, CalFADataFeb22.txt, and CalErr.m

This file is used by Fig3aAndS2.m

Note that the file location in lines 46, 54, and 59 should be added before
execution.

Version date m-file: Feb 22, 2023
%}

warning('off','MATLAB:table:ModifiedAndSavedVarnames')

% Some constants
Vm = 22.4;
FlowRate = 30/60;       % ml/s
F = 96485;              % Faraday constant
MwFA = 47;              % Formic acid molecular weight
Vcat = 50/1000;         % Catholyte/anolyte volume (L)

% Intercept and slope of calibration curve for H2
b0H2 = -2634.51869;
b1H2 = 307.16226;

% Intercept and slope of calibration curve for CO
b0CO = 4114.67521;
b1CO = 22.06563;

% Intercept and slope of calibration cure for FA (Feb22data)
b0FA = 16.94225;
b1FA = 473.16008;

%% Calibration input

% H2
CalH2Data = importdata('...\Matlab\CalH2Data.txt');
XCalH2 = [500*ones(12,1); 1000*ones(12,1); 1500*ones(12,1); 2500*ones(10,1); 5000*ones(12,1); 8000*ones(12,1); 10000*ones(12,1)]';
YCalH2 = [CalH2Data.data(:,1); CalH2Data.data(:,2); CalH2Data.data(:,3); CalH2Data.data(:,4); CalH2Data.data(:,5); CalH2Data.data(:,6); CalH2Data.data(:,7)]';

% CO
CalCOData = importdata('...\Matlab\CalCOData.txt');
XCalCO = [500*ones(12,1); 1000*ones(11,1); 1500*ones(12,1); 2500*ones(10,1); 5000*ones(12,1); 8000*ones(12,1)]';
YCalCO = [CalCOData.data(:,1); CalCOData.data(:,2); CalCOData.data(:,3); CalCOData.data(:,4); CalCOData.data(:,5); CalCOData.data(:,6)]';

% FA calibration Feb2022
CalFAFeb22Data = importdata('...\Matlab\CalFADataFeb22.txt');
XCalFAFeb22 = [10; 50; 100; 250]';
YCalFAFeb22 = [CalFAFeb22Data.data(:,1); CalFAFeb22Data.data(:,2); CalFAFeb22Data.data(:,3); CalFAFeb22Data.data(:,4)]';

%% Calculate total and partial current densities, working electrode potential (iR drop compensated)

for j = 1:size(VarName)

    % Retrieve file path for potentiostat data
    FilePathEC = append(Location, Folder1(j,:), FileNameEC(j,:));
    FilePathEC = cell2mat(FilePathEC);  % Convert cell array with file paths to matrix

    %% Load data from potentiostat
    LoadDataEC = importdata(FilePathEC);

    % LoadDataEC is a struct containing fields for every investigated potential
    % The columns refer to: time/s, <I>/mA, Ewe/V, <Ece>/V, dQ/C, Rcmp/Ohm,
    % P/W, Ewe-Ece/V
    Data.(VarName(j,:)).EC = LoadDataEC.data;

    % Retrieve Ewe
    Data.(VarName(j,:)).Ewe = Data.(VarName(j,:)).EC(:,3);

    % Recalculate time (time does not start at t=0)
    tElapsed = Data.(VarName(j,:)).EC(:,1);
    tStart = Data.(VarName(j,:)).EC(1,1);
    Data.(VarName(j,:)).t = tElapsed - tStart;

    % Calculate current density
    Data.(VarName(j,:)).CD = Data.(VarName(j,:)).EC(:,2)/Area(j);

    % Calculate average current density over the last 30 minutes of the
    % experiment
    findTime = find(Data.(VarName(j,:)).t >= 30*60);
    Data.(VarName(j,:)).CDavg30mins(1,1) = mean(Data.(VarName(j,:)).CD(findTime(:,1),1));

    % Calculate the error in the total current density
    Data.(VarName(j,:)).CDavg30mins(1,2) = sqrt(((AreaErr(j)/Area(j))^2))*Data.(VarName(j,:)).CDavg30mins(1,1);

    %% Retrieve file path for TCD data files to calculate H2 and CO partial current density
    FileNameTCD = {'\Anne20220208-TCD.txt'};
    FilePathTCD = append(Location, Folder1(j,:), FileNameTCD);
    FilePathTCD = cell2mat(FilePathTCD);  % Convert cell array with file paths to matrix

    %% Obtain H2 current and H2 partial current density
    LoadDataTCD = readtable(FilePathTCD);
    LoadDataTCD = LoadDataTCD(:,7:end);
    Data.(VarName(j,:)).TCD = LoadDataTCD;

    TCDArea = table2array(Data.(VarName(j,:)).TCD);

    % Calculate average area of injections 5, 6 and 7
    Data.TCD_avg(j,:) = mean(TCDArea(5:end,:));

    % Obtain H2 concentration and error in H2 concentration
    Width = size(Data.(VarName(j,:)).TCD(:,1));
    q = Width(1,1)-4; % Determine amount of values used to calculate the average
    Data.ConcAndErr(j,:) = CalErr(XCalH2, YCalH2, q,b0H2,b1H2,Data.TCD_avg(j,1));

    % Calculate corresponding current
    Data.IH2(j,:) = ((((Data.ConcAndErr(j,:))/10000/100)*(FlowRate/Vm))*-2*F);

    % Calculate corresponding current density
    Data.CDH2(j,1) = Data.IH2(j,1)/Area(j);

    % Calculate error in current density
    Data.CDH2(j,2) = sqrt(((Data.IH2(j,2))/Data.IH2(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.CDH2(j,1));

    %% Obtain CO current and CO partial current density

    % Obtain CO concentration and error in CO concentration
    Width = size(Data.(VarName(j,:)).TCD(:,4));
    qCO = Width(1,1)-4; % Determine amount of values used to calculate the average
    Data.ConcAndErrCO(j,:) = CalErr(XCalCO, YCalCO, qCO,b0CO,b1CO,Data.TCD_avg(j,4));

    % Calculate corresponding current
    Data.ICO(j,:) = ((((Data.ConcAndErrCO(j,:))/10000/100)*(FlowRate/Vm))*-2*F);

    % Calculate corresponding current density
    Data.CDCO(j,1) = Data.ICO(j,1)/Area(j);

    % Calculate error in current density
    Data.CDCO(j,2) = sqrt(((Data.ICO(j,2))/Data.ICO(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.ICO(j,1));

    %% Obtain FA current and FA partial current density if required (yes/no)
    if FA == 'y'

        % Retrieve file path of FA data
        FilePathFA = append(Location, '\CP FA areas.txt');
        FilePathFA = cell2mat(FilePathFA);  % Convert cell array with file paths to matrix
        LoadDataFA = importdata(FilePathFA);
        Data.FA(j,:) = LoadDataFA.data(j,:);

        % FA concentration in catholyte and corresponding error
        Data.ConcAndErrFACat(j,:) = CalErr(XCalFAFeb22,YCalFAFeb22,1,b0FA,b1FA,Data.FA(j,1));

        % FA concentration in anolyte and corresponding error
        Data.ConcAndErrFAAn(j,:) = CalErr(XCalFAFeb22,YCalFAFeb22,1,b0FA,b1FA,Data.FA(j,2));

        if Data.ConcAndErrFACat(j,1) > 0  & Data.ConcAndErrFAAn(j,1) > 0
            Data.ConcAndErrFA(j,1) = Data.ConcAndErrFACat(j,1) + Data.ConcAndErrFAAn(j,1);
            Data.ConcAndErrFA(j,2) = sqrt(Data.ConcAndErrFACat(j,2)^2 + Data.ConcAndErrFAAn(j,2)^2);
        elseif Data.ConcAndErrFACat(j,1) > 0
            Data.ConcAndErrFA(j,1) = Data.ConcAndErrFACat(j,1);
            Data.ConcAndErrFA(j,2) = Data.ConcAndErrFACat(j,2);

        else    % This means only Data.ConcAndErrFAAn(j,:) > 0
            Data.ConcAndErrFA(j,1) = Data.ConcAndErrFAAn(j,1);
            Data.ConcAndErrFA(j,2) = Data.ConcAndErrFAAn(j,2);

        end


        % Calculate corresponding current
        Data.IFA(j,:) = (((Data.ConcAndErrFA(j,:))/1000/MwFA)*Vcat*-2*F)/3600*1000;

        % Calculate corresponding current density
        Data.CDFA(j,1) = Data.IFA(j,1)/Area(j);

        % Calculate error in current density
        Data.xErrFA = (((Data.ConcAndErrFA(j,2))/1000/MwFA)*Vcat*-2*F)/3600*1000;

        Data.CDFA(j,2) = sqrt((Data.xErrFA/Data.IFA(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.CDFA(j,1));

    else
    end
    %% File path resistance values
    FilePathR =  append(Location,  RFileName);
    FilePathR = cell2mat(FilePathR);  % Convert cell array with file paths to matrix
    LoadDataResistance = importdata(FilePathR);

    %% Calculate actual potential at working electrode and error in the potential

    Data.(VarName(j,:)).R = LoadDataResistance.data(j,:);

    RAvg = mean(Data.(VarName(j,:)).R);

    % Actual potentail
    Data.(VarName(j,:)).Ereal = Data.(VarName(j,:)).Ewe - (Data.(VarName(j,:)).EC(:,2)/1000)*RAvg;

    % Error calculations
    RErr = Data.(VarName(j,:)).R(1,1) - RAvg;

    %Error in potential
    Data.(VarName(j,:)).Ereal(:,2) =  (Data.(VarName(j,:)).EC(:,2)/1000)*RErr;
end
end